import { Request, Response } from 'express';
import { consultationService } from './service';

export const consultationController = {
  // POST /consultations/forms
  async createForm(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { data, error } = await consultationService.createForm(vendor.id, req.body);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /consultations/forms
  async listForms(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { data, error } = await consultationService.getVendorForms(vendor.id);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /consultations/forms/:id
  async getForm(req: any, res: Response) {
    const { id } = req.params;
    const { data, error } = await consultationService.getFormWithQuestions(id);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // POST /consultations/forms/:form_id/respond
  async submitResponse(req: any, res: Response) {
    const { form_id } = req.params;
    const { data, error } = await consultationService.submitResponse(form_id, req.body);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /consultations/forms/:id/responses
  async getResponses(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { id } = req.params;
    const { customer_id, date_range } = req.query;
    
    const filters: any = {};
    if (customer_id) filters.customer_id = customer_id;
    if (date_range) filters.date_range = date_range;

    const { data, error } = await consultationService.getFormResponses(id, filters);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // GET /consultations/forms/:id/analytics
  async getAnalytics(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { id } = req.params;
    const { data, error } = await consultationService.getFormAnalytics(id);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // POST /consultations/forms/:id/duplicate
  async duplicateForm(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { id } = req.params;
    const { title } = req.body;
    
    if (!title) return res.status(400).json({ error: 'New title is required' });

    const { data, error } = await consultationService.duplicateForm(vendor.id, id, title);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  },

  // PATCH /consultations/forms/:id/status
  async updateFormStatus(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });

    const { id } = req.params;
    const { status } = req.body;
    
    if (!['draft', 'active', 'inactive'].includes(status)) {
      return res.status(400).json({ error: 'Invalid status' });
    }

    const { data, error } = await consultationService.updateFormStatus(id, status);
    if (error) return res.status(400).json({ error });
    return res.json({ success: true, data });
  }
}; 